<?php

namespace GoDaddy\WordPress\MWC\Core\Features\Commerce\Catalog\Providers\DataObjects;

use GoDaddy\WordPress\MWC\Core\Features\Commerce\Providers\DataObjects\AbstractDataObject;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Providers\DataObjects\ExternalId;
use GoDaddy\WordPress\MWC\Core\Features\Commerce\Providers\DataObjects\SimpleMoney;

/**
 * Product base data object.
 */
class ProductBase extends AbstractDataObject
{
    /** @var string */
    const CONDITION_NEW = 'NEW';

    /** @var string */
    const CONDITION_RECONDITIONED = 'RECONDITIONED';

    /** @var string */
    const CONDITION_REFURBISHED = 'REFURBISHED';

    /** @var string */
    const CONDITION_USED = 'USED';

    /** @var string digital product type (must have files to qualify) */
    const TYPE_DIGITAL = 'DIGITAL';

    /** @var string physical product type (default product type) */
    const TYPE_PHYSICAL = 'PHYSICAL';

    /** @var string service product type */
    const TYPE_SERVICE = 'SERVICE';

    /** @var string the default tax category */
    const TAX_CATEGORY_STANDARD = 'standard';

    /** @var bool a boolean flag indicating if the product can be sold (defaults to true) */
    public bool $active = true;

    /** @var bool|null whether the merchant can override the defined price */
    public ?bool $allowCustomPrice;

    /** @var string|null a separate unique identifier for this product like a slug */
    public ?string $altId = null;

    /** @var array<ImageAsset|VideoAsset>|null a collection of assets in the order they should be presented (the first assets is the default one) */
    public ?array $assets;

    /** @var string|null optional product brand */
    public ?string $brand;

    /** @var string[] */
    public array $categoryIds = [];

    /** @var ChannelIds|string[] */
    public $channelIds = [];

    /** @var string|null product condition - one of NEW, USED, RECONDITIONED, REFURBISHED */
    public ?string $condition = null;

    /** @var string|null datetime when the product was created (will be autogenerated if unspecified) */
    public ?string $createdAt;

    /** @var string|null datetime when the product was deleted */
    public ?string $deletedAt = null;

    /** @var string|null the product's description */
    public ?string $description;

    /** @var string|null the unique EAN (or UPC) value associated with the product (usually all digits) */
    public ?string $ean;

    /** @var ?ExternalId[] a collection of alternate identifiers for the product (such as GTIN or ASIN) */
    public ?array $externalIds;

    /** @var ?File[] used by digital products */
    public ?array $files;

    /** @var Inventory|null the product's inventory */
    public ?Inventory $inventory;

    /** @var ManufacturerData|null when present identifies product characteristics as declared by the manufacturer */
    public ?ManufacturerData $manufacturerData;

    /** @var Metadata[]|null commerce platform metadata */
    public ?array $metadata = null;

    /** @var string the product's name */
    public string $name;

    /** @var ?AbstractOption[] a collection of options that can be applied to the product, in the order they should be presented */
    public ?array $options;

    /** @var string|null primarily used by variants that share a parent product */
    public ?string $parentId;

    /** @var SimpleMoney|null the product price (required for products that do not have a {@see ProductBase::$parentId} */
    public ?SimpleMoney $price;

    /** @var string|null product UUID (assigned upon product creation) */
    public ?string $productId = null;

    /** @var bool|null whether the product is purchasable (defaults to true) */
    public ?bool $purchasable = true;

    /** @var SimpleMoney|null optional discounted price */
    public ?SimpleMoney $salePrice;

    /** @var ShippingWeightAndDimensions|null optional product weight and dimensions (if applicable) */
    public ?ShippingWeightAndDimensions $shippingWeightAndDimensions;

    /** @var string|null short identifier for merchant-facing visualization on some devices (e.g. Poynt terminal) */
    public ?string $shortCode;

    /** @var string SKU (required) */
    public string $sku;

    /** @var string|null optional tax category under which the product falls */
    public ?string $taxCategory;

    /** @var string describes the type of product */
    public string $type = self::TYPE_PHYSICAL;

    /** @var string|null datetime when the product was last modified (will be autogenerated if unspecified) */
    public ?string $updatedAt;

    /** @var VariantOptionMapping[]|null list of {@see VariantListOption} mapped to a product variant */
    public ?array $variantOptionMapping = null;

    /** @var string[]|null list of variants productIds belonging to this product */
    public ?array $variants = null;

    /**
     * Product data object constructor.
     *
     * @param array{
     *     active: bool,
     *     allowCustomPrice: ?bool,
     *     altId?: ?string,
     *     assets: ?array<AbstractAsset|ImageAsset|VideoAsset>,
     *     brand: ?string,
     *     categoryIds: string[],
     *     channelIds: ChannelIds|string[],
     *     condition?: ?string,
     *     createdAt: ?string,
     *     deletedAt?: ?string,
     *     description: ?string,
     *     ean: ?string,
     *     externalIds: ?ExternalId[],
     *     files?: ?File[],
     *     inventory: ?Inventory,
     *     manufacturerData: ?ManufacturerData,
     *     metadata?: ?Metadata[],
     *     name: string,
     *     options: ?AbstractOption[],
     *     parentId: ?string,
     *     price: ?SimpleMoney,
     *     productId: ?string,
     *     purchasable?: ?bool,
     *     salePrice: ?SimpleMoney,
     *     shippingWeightAndDimensions: ?ShippingWeightAndDimensions,
     *     shortCode: ?string,
     *     sku: string,
     *     taxCategory: ?string,
     *     type: ?string,
     *     updatedAt: ?string,
     *     variantOptionMapping?: ?VariantOptionMapping[],
     *     variants?: ?string[],
     * } $data
     */
    public function __construct(array $data)
    {
        parent::__construct($data);
    }
}
